function [autov, jauto, x, autof] = disparo_autovalores(xa, xb, mut, ...
                                                            nauto)
%.
% [autov, jauto, x, autof] = disparo_autovalores(xa,xb,mut,nauto)
%
% Devuelve los autovalores y las autofunciones para la parte espacial del 
% problema de la cuerda vibrante, calculados por un metodo de disparo.
%
% ENTRADA
%
% xa,xb    limites inferior y superior del dominio de integracion
% mut      masa/tension (funcion de x)
% nauto    numero de autovalores que se desea obtener
%
% SALIDA
%
% autov    array de nauto elementos con los autovalores encontrados
% jauto    numero de autovalores hallados
% autof    matriz en que cada columna contiene los valores de una
%          autofuncion evaluada en los puntos del array x
% x        array de valores de x en los que se devuelve la autofuncion
%

% PARAMETROS DE LA EXPLORACION NUMERICA
da = 1.0; % paso con el cual se buscan los autovalores
nx = 50;   % numero de puntos en los que se van a dar valores a las 
           % autofunciones
aamax = 1.e3; %maximo valor de busqueda de los autovalores 
              % ESTO ES BASTANTE ARBITRARIO
                      
x  = linspace( xa, xb, nx);  % malla en variable independiente
autov = zeros( nauto, 1);    % inicializamos el array de autovalores

%
% Redefinimos la función 'ode_cv' para que sea utilizable por la rutina
% 'ode45'. Para ello, tenemos que eliminar la dependencia en 'mut'. Lo
% fijamos mediante una función anonima.
%
% Redefinimos la funcion de integracion de forma que pueda ser usada por 
% 'fzero', que admite funciones de una sola variable. Para ello, 
% redefinimos 'int_ode_cv' mediante una funcion anonima para que solo
% dependa del parametro 'aa' (fijando el resto).
%
fode = @(x,y) ode_cv(x,y,mut);
fdis = @(t) int_ode_cv( fode, [xa xb], [0.0 1.0 t], 0.0);

%
% Para buscar los ceros de la funcion 'fdis' vamos recorriendo valores 
% del parametro 'aa' en saltos de valor 'da'. Cada vez que detectemos
% un cambio de signo de la funcion 'fdes' buscamos un cero con la funcion
% 'fzero' y guardamos el resultado en el array autov.
%
s = sign( fdis( da ));
aa = 2*da;
jauto = 0;
figure(10);clf;hold on;
while ( ( aa  < aamax  ) && (jauto < nauto) )
    ycc = fdis(aa);
    %plot(aa,ycc,'o');pause(0.005);
    if( sign(ycc) ~= s )
        s = -s;
        jauto = jauto + 1;
        autov(jauto) = fzero( fdis, [aa-da aa] );
        fprintf('\n Autovalor encontrado %e\n',autov(jauto));
    end
    aa = aa + da;
end

%
% Una vez hallados los autovalores, hallamos las autofunciones mediante
% integracion directa. Guardamos el resultado en la matriz autof.
%

autof = zeros( length(x), jauto);
for ii=1:jauto
    [x, yy ] = ode45( fode, x, [0.0 1.0 autov(ii)]);
    autof( :, ii) = yy(:, 1)/max( yy(:,1) );
end

function [ycc] = int_ode_cv(odefun,tspan,y0,yb)
%
% [ycc] = int_ode_cv(odefun,tspan,y0)
%
% Devuelve el 'desvio' del disparo.  Integra el sistema de ODEs descrito 
% por odefun. Al valor de la integral en el limite superior de dominio 
% le resta el valor objetivo. Hace uso de la funcion ODE45
%
% ENTRADA
%   odefun, tspan, y0  mismo significado que en ODE45
%   yb                 valor de la variable dependiente en el extremo
%                      superior del dominio (condicion de contorno).
% SALIDA
%   ycc                diferencia entre y(x maximo) - yb.
%
options = odeset('RelTol',1e-5);
[x, y] = ode45( odefun, tspan, y0, options);
ycc = y(end,1)-yb;

function [yy]=ode_cv(x,y,mut)
%
% [yy]=ode_cv(x,y,mut)
%
% Devuelve las derivadas correspondientes a la parte espacial de la
% ecuacion de una cuerda vibrante en la forma y'=f(x,y) (y,y' arrays
% de TRES elementos). Se incluye al autovalor como una variable y(3),
% con ecuacion diferencial y'(3) = 0.0. Entonces y(3)=omega^2.
%
% ENTRADA
%   x     valor de la variable independiente en el que se calculan las 
%         derivadas (escalar)
%   y     valor de la variable dependiente en el que se calculas las 
%         derivadas (array de dos elementos)
% mut     masa de la cuerda entre tension (funcion)
%
% SALIDA
%   yy     yy = y' (vector columna)
%
yy = [ y(2); -mut(x)*y(3)*y(1) ; 0.0];
